import re
import ipaddress
from typing import Dict, List, Set, Tuple
from collections import defaultdict

class FortiGateIPMatcher:
    def __init__(self):
        self.address_objects: Dict[str, List[ipaddress.IPv4Network]] = {}
        self.total_objects = 0

    def parse_config_file(self, config_file_path: str) -> None:
        """Parse FortiGate config file and extract address objects."""
        current_object = None
        with open(config_file_path, 'r') as f:
            for line in f:
                line = line.strip()
                
                # Check for address object definition
                if line.startswith('edit '):
                    current_object = line.split('"')[1] if '"' in line else line.split()[1]
                    self.address_objects[current_object] = []
                    self.total_objects += 1
                
                # Extract subnet information
                elif current_object and ('set subnet' in line):
                    # Extract IP and mask
                    subnet_match = re.search(r'set subnet (\d+\.\d+\.\d+\.\d+) (\d+\.\d+\.\d+\.\d+)', line)
                    if subnet_match:
                        ip, mask = subnet_match.groups()
                        try:
                            network = ipaddress.IPv4Network(f'{ip}/{mask}', strict=False)
                            self.address_objects[current_object].append(network)
                        except ValueError:
                            print(f"Warning: Invalid subnet for {current_object}: {ip}/{mask}")

                elif line.startswith('next'):
                    current_object = None

    def match_ips(self, ip_list: List[str]) -> Dict[str, List[str]]:
        """Match IPs against address objects and return matches grouped by address object."""
        results = defaultdict(list)
        unmatched_ips = []
        invalid_ips = []
        
        # Track which IPs match multiple objects
        multiple_matches = defaultdict(set)
        
        for ip_str in ip_list:
            try:
                # Convert IP string to IP address object
                ip = ipaddress.IPv4Address(ip_str)
                ip_matched = False
                match_count = 0
                
                # Check against all address objects
                for obj_name, networks in self.address_objects.items():
                    for network in networks:
                        if ip in network:
                            results[obj_name].append(ip_str)
                            ip_matched = True
                            match_count += 1
                            if match_count > 1:
                                multiple_matches[ip_str].add(obj_name)
                
                if not ip_matched:
                    unmatched_ips.append(ip_str)
                    
            except ValueError:
                invalid_ips.append(ip_str)
        
        return dict(results), unmatched_ips, invalid_ips, multiple_matches

def format_ip_list(ips: List[str], max_display: int = 10) -> str:
    """Format IP list with optional truncation."""
    if len(ips) <= max_display:
        return "\n    " + "\n    ".join(sorted(ips))
    else:
        return f"\n    " + "\n    ".join(sorted(ips)[:max_display]) + f"\n    ... ({len(ips) - max_display} more)"

def main():
    matcher = FortiGateIPMatcher()
    
    # Get input file paths from user
    config_file = input("Enter the path to FortiGate config file: ")
    ip_list_file = input("Enter the path to IP list file (one IP per line): ")
    
    # Read IP list
    with open(ip_list_file, 'r') as f:
        ip_list = [line.strip() for line in f if line.strip()]
    
    print(f"\nProcessing {len(ip_list)} IPs against FortiGate configuration...")
    
    # Parse config and match IPs
    matcher.parse_config_file(config_file)
    results, unmatched, invalid, multiple_matches = matcher.match_ips(ip_list)
    
    # Output results
    print(f"\nFound matches in {len(results)} out of {matcher.total_objects} address objects")
    print("-" * 60)
    
    if results:
        # Sort address objects by number of matching IPs (descending)
        sorted_objects = sorted(results.items(), key=lambda x: len(x[1]), reverse=True)
        
        for obj_name, ips in sorted_objects:
            print(f"\nAddress Object: {obj_name}")
            print(f"Matching IPs ({len(ips)}):{format_ip_list(ips)}")
    
    if multiple_matches:
        print("\nIPs matching multiple address objects:")
        print("-" * 60)
        for ip, objects in multiple_matches.items():
            print(f"  {ip} matches:")
            for obj in sorted(objects):
                print(f"    - {obj}")
    
    if unmatched:
        print("\nUnmatched IPs:")
        print("-" * 60)
        print(f"Count: {len(unmatched)}{format_ip_list(unmatched)}")
    
    if invalid:
        print("\nInvalid IP formats:")
        print("-" * 60)
        print(format_ip_list(invalid))
            
    # Print summary
    print("\nSummary:")
    print("-" * 60)
    print(f"Total address objects in config: {matcher.total_objects}")
    print(f"Address objects with matches: {len(results)}")
    print(f"Total IPs processed: {len(ip_list)}")
    print(f"IPs matched to address objects: {sum(len(ips) for ips in results.values())}")
    print(f"IPs matching multiple objects: {len(multiple_matches)}")
    print(f"Unmatched IPs: {len(unmatched)}")
    print(f"Invalid IPs: {len(invalid)}")
    
if __name__ == "__main__":
    main()
